<?php 
include 'config.php';
session_start();


/**
 * get the API.
 * This function will return the API for a specific utility. 
 * 
 * @param string $utility utility of the API. EX: If your utility is to load doctors, 
 *                        then pass load_doctor as utility and the function will give you the corresponding API.
 * @return string|boolean returns the API, if found your utility else FALSE
 */
function getAPI($utility) {
$api = array(
        "get_op_feedback_form" => "get-op-feedback-form",
        "save_op_feedback_form" => "save-op-feedback-form",
    );
    if (isset($api[$utility]) && !empty($api[$utility])) {
        return $api[$utility];
    }
    return FALSE;
}

/**
 * Makes a curl request.
 * 
 * @param type STRING request method like POST, GET, PUT, DELETE etc
 * @param type STRING request body content.
 * @param type STRING access token for granting access to the api.
 * @param type STRING api path.
 * @return array return the api data in JSON array
 */
function curlForAPI($method, $content, $access_token, $path) {
    
    $ch = curl_init();
    $headers = ["Content-type: application/json", "Accept: application/json, application/pdf", "Authorization: Bearer " . $access_token];
    curl_setopt($ch, CURLOPT_URL, $path);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

    if ("POST" === $method) {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $content);
    }

    $response = curl_exec($ch);
    $data = json_decode($response, true);
    curl_close($ch);
    return $data;
}


/**
 * get the access token
 * 
 * @return string returns the access token
 */
function getAccessToken() {

//The generated access token is stored in session for later usage. Let's check if it's still there.
    if (isset($_SESSION['access_token']) && !empty($_SESSION['access_token'])) {
//Yep, it's still there.
//So lets check if the access token has expired.
//Access token will expire after 3600 seconds.
//What we actually do here to check access token expiry is that, we will store the time of access token generation in session named $_SESSION['time_of_access_token_generation'],
//so if the difference between $_SESSION['time_of_access_token_generation'] and the current timestamp is greatherthan 3600, that means, the access token has expired.
//In that case request for a new access token.
        if (isset($_SESSION['time_of_access_token_generation']) && (time() - $_SESSION['time_of_access_token_generation'] > 3600)) {
//get the new access token.
            setAccessToken();
        }
    } else {
//In case of session expiry, request for a new access token.
        setAccessToken();
    }

//Get the access token from session.
    $access_token = isset($_SESSION['access_token']) ? $_SESSION['access_token'] : NULL;

//by now, we have our access token, so let's use it.
    return $access_token;
}


/**
 * Generate access token and store it in session.
 * 
 * @return boolean TRUE
 */
function setAccessToken() {

    //Request for new access token
    $path = REST_API_ROOT .'web-site-login';
    $content = array('data' => array(
            'grant_type' => "client_credentials",
            'client_id' => CLIENT_ID,
            'client_secret' => CLIENT_SECRET
    ));
    $data = curlForCreateToken($path, 'POST', json_encode($content));
    $access_token = isset($data['access_token']) && !empty($data['access_token']) ? $data['access_token'] : NULL;
//store the access token in session.
    $_SESSION['access_token'] = $access_token;
//store the time of access token generation in session.
    $_SESSION['time_of_access_token_generation'] = time();

    return TRUE;
}
/**
 * Makes a curl request for getting access token
 * 
 * @return array returns an array containing access token details.
 */
function curlForCreateToken($path, $method, $content) {
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $path);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-type: application/json'));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

    if ("POST" === $method) {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $content);
    }
    $response = curl_exec($ch);
   
    if (curl_errno($ch)) {
        $error_msg = curl_error($ch);
        $data = array('status' => 'error', 'msg' => $error_msg);
        return json_decode($data);
    }
    $data = json_decode($response, true);
    curl_close($ch);
    if ("error" == $data['status']) {
        return $data;
    } else {
        $data = $data['responseArray'];
    }
    return $data;
}


/**
 * encrypt data using the public key
 * @param type $data
 * @return type
 */
function encryptData($data) {
    $keyFile ='uploads/public.pem';
    $myfile = fopen($keyFile, "r") or die("Unable to open file!");
    $public_key = fread($myfile, filesize($keyFile));
    fclose($myfile);
    $key = null;
    $output = openssl_public_encrypt($data, $key, $public_key, OPENSSL_PKCS1_OAEP_PADDING);
    return base64_encode($key);
}


/**
 * decrypt data using the private key
 * @param type $data
 * @return type
 */
function decryptData($data) {
    $keyFile = 'uploads/private.pem';
    $myfile = fopen($keyFile, "r") or die("Unable to open file!");
    $private_key = fread($myfile, filesize($keyFile));
    fclose($myfile);

    $key = null;
    $output = openssl_private_decrypt(base64_decode($data), $key, $private_key, OPENSSL_PKCS1_OAEP_PADDING);
    return $key;
}





            

